package com.example.gemma

import android.os.Bundle
import android.text.method.ScrollingMovementMethod
import android.widget.*
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.lifecycleScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

class MainActivity : AppCompatActivity() {

    private lateinit var txtOutput: TextView
    private lateinit var edtPrompt: EditText
    private lateinit var btnSend: Button
    private lateinit var radioRoute: RadioGroup
    private lateinit var btnDownload: Button

    private val modelManager by lazy { ModelManager(this) }
    private var local: LocalGemma? = null
    private val cloud = CloudClient(baseUrl = "https://YOUR_CLOUD_ENDPOINT") // TODO: replace

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        if (!LicenseGate.hasAccepted(this)) {
            LicenseGate.show(this) {
                // user accepted
                setupUi()
            }
        } else {
            setupUi()
        }
    }

    private fun setupUi() {
        setContentView(R.layout.activity_main)

        txtOutput = findViewById(R.id.txtOutput)
        edtPrompt = findViewById(R.id.edtPrompt)
        btnSend = findViewById(R.id.btnSend)
        radioRoute = findViewById(R.id.radioRoute)
        btnDownload = findViewById(R.id.btnDownload)

        txtOutput.movementMethod = ScrollingMovementMethod()

        btnDownload.setOnClickListener {
            lifecycleScope.launch {
                appendLine("Downloading model…")
                val path = withContext(Dispatchers.IO) { modelManager.ensureModel() }
                appendLine("Model ready at: $path")
                loadLocal(path)
            }
        }

        btnSend.setOnClickListener {
            val prompt = edtPrompt.text.toString().trim()
            if (prompt.isEmpty()) return@setOnClickListener
            lifecycleScope.launch { routeAndGenerate(prompt) }
        }

        // Autoload if already present
        lifecycleScope.launch {
            modelManager.getLocalModelPath()?.let { loadLocal(it) }
        }
    }

    private suspend fun loadLocal(path: String) = withContext(Dispatchers.IO) {
        try {
            local?.close()
            local = LocalGemma(this@MainActivity, path).also { it.load() }
            appendLine("Local LLM loaded.")
        } catch (e: Exception) {
            appendLine("Failed to init local LLM: ${e.message}")
        }
    }

    private suspend fun routeAndGenerate(prompt: String) = withContext(Dispatchers.IO) {
        val routing = when (radioRoute.checkedRadioButtonId) {
            R.id.optLocal -> Routing.LOCAL_ONLY
            R.id.optCloud -> Routing.CLOUD_ONLY
            else -> Routing.AUTO
        }
        val router = Router(local, cloud, maxLocalInputTokens = 2048)
        appendLine("Routing: $routing")
        try {
            router.generate(prompt, routing, onToken = { token ->
                runOnUiThread { txtOutput.append(token) }
            }, onDone = { ok, source ->
                runOnUiThread { appendLine("\n\n[done: $ok via $source]") }
            })
        } catch (e: Exception) {
            appendLine("Error: ${e.message}")
        }
    }

    private fun appendLine(msg: String) = runOnUiThread {
        txtOutput.append("\n$msg")
    }

    override fun onDestroy() {
        super.onDestroy()
        local?.close()
    }
}
