import Foundation

struct CloudClient {
    let baseUrl: String

    func generate(prompt: String) async throws -> String {
        guard let u = URL(string: "\(baseUrl.trimmingCharacters(in: CharacterSet(charactersIn: "/")))/generate?prompt=\(prompt.addingPercentEncoding(withAllowedCharacters: .urlQueryAllowed) ?? "")")
        else { throw URLError(.badURL) }
        let (data, _) = try await URLSession.shared.data(from: u)
        return String(data: data, encoding: .utf8) ?? ""
    }
}

enum Routing { case auto, localOnly, cloudOnly }

@MainActor
struct Router {
    let local: LocalLlm?
    let cloud: CloudClient?
    let maxLocalInputTokens: Int

    private func estimateTokens(_ s: String) -> Int { max(1, s.count / 4) }

    func generate(prompt: String, routing: Routing,
                  onToken: @escaping (String) -> Void,
                  onDone: @escaping (Bool, String) -> Void) async throws {
        switch routing {
        case .localOnly:
            guard let local else { onDone(false, "local-unavailable"); return }
            try await local.generateStream(prompt: prompt, onToken: onToken) { ok in onDone(ok, "local") }
        case .cloudOnly:
            let out = try await cloud?.generate(prompt: prompt) ?? "[cloud unavailable]"
            onToken(out); onDone(true, "cloud")
        case .auto:
            let tokens = estimateTokens(prompt)
            if let local, tokens <= maxLocalInputTokens {
                try await local.generateStream(prompt: prompt, onToken: onToken) { ok in onDone(ok, "local") }
            } else {
                let out = try await cloud?.generate(prompt: prompt) ?? "[cloud unavailable]"
                onToken(out); onDone(true, "cloud")
            }
        }
    }
}
