import SwiftUI

struct ContentView: View {
    @State private var accepted = UserDefaults.standard.bool(forKey: "gemma_terms_accepted")
    @State private var prompt: String = ""
    @State private var output: String = "Ready."
    @State private var useRoute: Int = 0 // 0 = Auto, 1 = Local, 2 = Cloud
    @StateObject private var local = LocalLlm()
    private let cloud = CloudClient(baseUrl: "https://YOUR_CLOUD_ENDPOINT") // TODO

    var body: some View {
        VStack(alignment: .leading, spacing: 12) {
            HStack {
                Button("Download / Load Model") {
                    output.append("\nDownloading model…")
                    Task {
                        let path = try? await ModelManager.shared.ensureModel()
                        output.append("\nModel ready at: \(path ?? "n/a")")
                        if let p = path {
                            try? await local.load(modelPath: p)
                            output.append("\nLocal LLM loaded.")
                        }
                    }
                }
                Spacer()
            }
            TextEditor(text: $prompt).frame(height: 120).border(.secondary)

            Picker("", selection: $useRoute) {
                Text("Auto").tag(0)
                Text("Local").tag(1)
                Text("Cloud").tag(2)
            }.pickerStyle(.segmented)

            Button("Send") {
                output.append("\nRouting…")
                Task {
                    await routeAndGenerate()
                }
            }

            ScrollView { Text(output).font(.system(size: 12, design: .monospaced))
                .frame(maxWidth: .infinity, alignment: .leading) }
        }
        .padding()
        .sheet(isPresented: .constant(!accepted)) {
            TermsSheet(accepted: $accepted)
        }
        .onChange(of: accepted) { _, v in
            UserDefaults.standard.set(v, forKey: "gemma_terms_accepted")
        }
    }

    func routeAndGenerate() async {
        let router = Router(local: local, cloud: cloud, maxLocalInputTokens: 2048)
        let routing: Routing = [Routing.auto, .localOnly, .cloudOnly][useRoute]
        do {
            output.append("\n---\n")
            try await router.generate(prompt: prompt, routing: routing,
                                      onToken: { token in
                                          output.append(token)
                                      }, onDone: { ok, src in
                                          output.append("\n\n[done: \(ok) via \(src)]")
                                      })
        } catch {
            output.append("\nError: \(error.localizedDescription)")
        }
    }
}

struct TermsSheet: View {
    @Binding var accepted: Bool
    var body: some View {
        VStack(spacing: 12) {
            Text("Gemma Terms of Use").font(.title3).bold()
            Text("You must accept the Gemma Terms of Use to download and run the model.")
            HStack {
                Link("View Terms", destination: URL(string: "https://ai.google.dev/gemma/terms")!)
                Spacer()
                Button("I Accept") { accepted = true }
            }
        }.padding()
    }
}
